<?php
/**
 * @package   MailChimp Discount
 * @author    MagniGenie
 * @link      https://magnigenie.com/?utm_source=mailchimp_discount&utm_medium=source_code
 * @copyright MagniGenie
 */

class Wcmd_Options_Framework_Admin {

	/**
	* Page hook for the options screen
	*
	* @since 1.7.0
	* @type string
	*/
  
  protected $options_screen = null;

	/**
	* Hook in the scripts and styles
	*
	* @since 1.7.0
	*/
  public function init() {

		// Gets options to load
    $options = & Wcmd_Options_Framework::_wcmd_optionsframework_options();

		// Checks if options are available
    if ( $options ) {

			// Add the options page and menu item.
			add_action( 'admin_menu', array( $this, 'add_custom_options_page' ) );

			// Add the required scripts and styles
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_styles' ) );

			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );

			// Settings need to be registered after admin_init
			add_action( 'admin_init', array( $this, 'settings_init' ) );

			// Adds options menu to the admin bar
			add_action( 'wp_before_admin_bar_render', array( $this, 'wcmd_optionsframework_admin_bar' ) );

			add_action( 'wp_ajax_wcmd_ajax_products', array( $this, 'wcmd_ajax_products' ) );
		}
  }

 public function wcmd_ajax_products() {
    	global $wpdb;
        $post_types = array( 'product', 'product_variation' );
        ob_start();

        if ( empty( $term ) ) {
            $term = wc_clean( stripslashes( $_GET['q'] ) );
        } else {
            $term = wc_clean( $term );
        }


        if ( empty( $term ) ) {
            die();
        }

        $like_term = '%' . $wpdb->esc_like( $term ) . '%';

        if ( is_numeric( $term ) ) {
            $query = $wpdb->prepare( "
                SELECT ID FROM {$wpdb->posts} posts LEFT JOIN {$wpdb->postmeta} postmeta ON posts.ID = postmeta.post_id
                WHERE posts.post_status = 'publish'
                AND (
                    posts.post_parent = %s
                    OR posts.ID = %s
                    OR posts.post_title LIKE %s
                    OR (
                        postmeta.meta_key = '_sku' AND postmeta.meta_value LIKE %s
                    )
                )
            ", $term, $term, $term, $like_term );
        } else {
            $query = $wpdb->prepare( "
                SELECT ID FROM {$wpdb->posts} posts LEFT JOIN {$wpdb->postmeta} postmeta ON posts.ID = postmeta.post_id
                WHERE posts.post_status = 'publish'
                AND (
                    posts.post_title LIKE %s
                    or posts.post_content LIKE %s
                    OR (
                        postmeta.meta_key = '_sku' AND postmeta.meta_value LIKE %s
                    )
                )
            ", $like_term, $like_term, $like_term );
        }

        $query .= " AND posts.post_type IN ('" . implode( "','", array_map( 'esc_sql', $post_types ) ) . "')";

        if ( ! empty( $_GET['exclude'] ) ) {
            $query .= " AND posts.ID NOT IN (" . implode( ',', array_map( 'intval', explode( ',', $_GET['exclude'] ) ) ) . ")";
        }

        if ( ! empty( $_GET['include'] ) ) {
            $query .= " AND posts.ID IN (" . implode( ',', array_map( 'intval', explode( ',', $_GET['include'] ) ) ) . ")";
        }

        if ( ! empty( $_GET['limit'] ) ) {
            $query .= " LIMIT " . intval( $_GET['limit'] );
        }

        $posts          = array_unique( $wpdb->get_col( $query ) );
        $found_products = array();

        if ( ! empty( $posts ) ) {
            foreach ( $posts as $post ) {
                $product = wc_get_product( $post );

                if ( ! current_user_can( 'read_product', $post ) ) {
                    continue;
                }

                if ( ! $product || ( $product->is_type( 'variation' ) && empty( $product->parent ) ) ) {
                    continue;
                }

                $found_products[ $post ] = str_replace('&#8211;', '-', get_the_title ( $product->get_id() )) . ' (#' .  $product->get_id() .')' ;
            }
        }
        // print_r($found_products);
        // exit;
        wp_send_json( $found_products );
    }



  public function add_tabs() {
    $screen = get_current_screen();

    if ( $screen->id != 
      'toplevel_page_woocommerce-mailchimp-discount' )
      return;
    
    $screen->add_help_tab(
      array(
        'id'      => 'wcmd_wpml',
        'title'   => __( 'WPML ShortCodes ', 'wcmd' ),
        'content' =>
          '<p>' . __( 'You can use [wcmd] shortcode to translate the contents of email body, email subject, popup text, popup header text and success message. Please find the list of variables.', 'woocommerce' ) . '</p>' .

          '<table class="widefat">
            <tr>
              <th>Variable</th>
              <th>Description</th>
            </tr>
            <tr>
              <td>lang</td>
              <td>Language code of the content</td>
            </tr>
          </table>' .
          '<p>' . __( 'Here some examples below:</br>[wcmd lang="en"]English Content[/wcmd]</br>[wcmd lang="fr"]French Content[/wcmd]', 'woocommerce' ) . '</p>'
      )
    );

    $screen->add_help_tab( 
      array(
        'id'      => 'wcmd_help',
        'title'   => __( 'Mailchimp Discount ', 'wcmd' ),
        'content' =>
          '<p>' . __( 'Thanks for purchasing the plugin. Please find the list of variables you can use for email body and email subject.', 'woocommerce' ) . '</p>' .

          '<table class="widefat">
            <tr>
              <th>Variable</th>
              <th>Description</th>
            </tr>
            <tr>
              <td><input readonly value="{COUPONCODE}"></td>
              <td>The coupon code which the user will use to reedem his discount. Make sure you have added this in email content otherwise the user can\'t get the discount.</td>
            </tr>
            <tr>
              <td><input readonly value="{COUPONEXPIRY}"></td>
              <td>It will output the coupon expiry date if you have entered a value for coupon validity.</td>
            </tr>
            <tr>
              <td><input readonly size="26" value="{ALLOWEDCATEGORIES}"></td>
              <td>It will display the list of categories with their link on which the discount is applicable. Make sure you have selected some categories otherwise it will output nothing.</td>
            </tr>
            <tr>
              <td><input readonly size="26" value="{EXCLUDEDCATEGORIES}"></td>
              <td>It will display the list of categories with their link on which the discount is not applicable. Make sure you have selected some categories otherwise it will output nothing.</td>
            </tr>
            <tr>
              <td><input readonly size="26" value="{ALLOWEDPRODUCTS}"></td>
              <td>It will display the list of products with their link on which the discount is applicable. Make sure you have selected some products otherwise it will output nothing.</td>
            </tr>
            <tr>
              <td><input readonly size="26" value="{EXCLUDEDPRODUCTS}"></td>
              <td>It will display the list of products with their link on which the discount is not applicable. Make sure you have selected some products otherwise it will output nothing.</td>
            </tr>
            <tr>
              <td><input readonly size="26" value="{EmailAddress}"></td>
              <td>It will display the email address. To be use only in email message.</td>
            </tr>
            <tr>
              <td><input readonly size="26" value="{FirstName}"></td>
              <td>It will display the First Name. To be use only in email message.</td>
            </tr>
            <tr>
              <td><input readonly size="26" value="{LastName}"></td>
              <td>It will display the Last Name. To be use only in email message.</td>
            </tr>
          </table>'
      )
    );

    $screen->add_help_tab(
      array(
        'id'      => 'wcmd_help_shortcode',
        'title'   => __( 'Mailchimp Discount Shortcode', 'wcmd' ),
        'content' => 
          '<p>' . __( 'You can use <i>[wc_mailchimp_discount]</i> shortcode to use the mailchimp discout form on your page/post/widget etc.<br>Please find the list of variables you can use with shortcode.' ) . '</p>'.

          '<table class="widefat">
            <tr>
              <th>Variable</th>
              <th>Description</th>
            </tr>
            <tr>
              <td><input readonly size="10" value="width"></td>
              <td>Define a width for the signup form. <br>Possible values: 100px, 100%, 500px etc. <br>Usage: [wc_mailchimp_discount width="400px"]</td>
            </tr>
            <tr>
              <td><input readonly size="10" value="align"></td>
              <td>Set the alignment for the signup form. <br> Possible values: left,right and center.<br>Usage: [wc_mailchimp_discount align="center"]</td>
            </tr>
            <tr>
              <td><input readonly size="10" value="btn_width"></td>
              <td>Set width for the subscribe button.<br> Possible values: 100px, 429px, 100%, 69% etc.<br>Usage: [wc_mailchimp_discount btn_width="300px"]</td>
            </tr>
            <tr>
              <td><input readonly size="10" value="btn_align"></td>
              <td>Set the alignment for the subscribe button. <br> Possible values: left,right and center.<br>Usage: [wc_mailchimp_discount btn_align="right"]</td>
            </tr>
            <tr>
              <td><input readonly size="10" value="top_text"></td>
              <td>Define a text that would appear on top of the form.<br>Usage: [wc_mailchimp_discount top_text="Subscribe to our newsletter and win discount"]</td>
            </tr>
            <tr>
              <td><input readonly size="16" value="top_text_color"></td>
              <td>Set a text color for the top text.<br>Usage: [wc_mailchimp_discount top_text_color="#ffcc00"]</td>
            </tr>
          </table>'.
          '<p>' . 'You can combine any of the shortcode variables and create different type of forms. Check some examples below:<br>'.
          '[wc_mailchimp_discount width="400px" align="center" btn_width="100%" texttop_text_top="Signup for newsletter" top_text_color="#333333"]</b>'
      )
    );
  }

	/**
	* Registers the settings
	*
	* @since 1.7.0
	*/
  function settings_init() {

  	// Load Options Framework Settings
    $wcmd_optionsframework_settings = get_option( 'wcmd_optionsframework' );


		// Registers the settings fields and callback
		register_setting( 'wcmd_optionsframework', $wcmd_optionsframework_settings['id'],  array ( $this, 'validate_options' ) );

		// Displays notice after options save
		add_action( 'wcmd_optionsframework_after_validate', array( $this, 'save_options_notice' ) );

  }

	/*
	* Define menu options
	*
	* Examples usage:
	*
	* add_filter( 'wcmd_optionsframework_menu', function( $menu ) {
	*     $menu['page_title'] = 'The Options';
	*	   $menu['menu_title'] = 'The Options';
	*     return $menu;
	* });
	*
	* @since 1.7.0
	*
	*/
	static function menu_settings() {

		$menu = array(

			// Modes: submenu, menu
	    'mode' => 'menu',

	    // Submenu default settings
	    'page_title' 	=> __( 'Theme Options', 'wc_mailchimp_discount'),
			'menu_title' 	=> __('Theme Options', 'wc_mailchimp_discount'),
			'capability' 	=> 'edit_theme_options',
			'menu_slug' 	=> 'options-framework',
	    'parent_slug' => 'themes.php',

	    // Menu default settings
	    'menu_icon' 		=> 'dashicons-menu',
	    'position' 		=> '61'
		);

		return apply_filters( 'wcmd_optionsframework_menu', $menu );
	}

	/**
	* Add a subpage called "Theme Options" to the appearance menu.
	*
	* @since 1.7.0
	*/
	function add_custom_options_page() {

		$menu = $this->menu_settings();

    	switch( $menu['mode'] ) {

      	case 'menu':
        
        // http://codex.wordpress.org/Function_Reference/add_menu_page
				$this->options_screen = add_menu_page(
					$menu['page_title'],
					$menu['menu_title'],
					$menu['capability'],
					$menu['menu_slug'],
					array( $this, 'options_page' ),
					$menu['menu_icon'],
					$menu['position']
				);
        break;

        default:
            	
        // http://codex.wordpress.org/Function_Reference/add_submenu_page
        $this->options_screen = add_submenu_page(
        	$menu['parent_slug'],
        	$menu['page_title'],
        	$menu['menu_title'],
        	$menu['capability'],
        	$menu['menu_slug'],
        	array( $this, 'options_page' ) );
        break;
      }

      add_action('load-'.$this->options_screen, array( $this, 'add_tabs' ) );
  
	}

  //add_action( 'admin_menu', 'wcmd_addons_menu', 100 );

  


	/**
	* Loads the required stylesheets
	*
	* @since 1.7.0
	*/

	function enqueue_admin_styles( $hook ) {

		if ( $this->options_screen != $hook )
	  	return;

			wp_enqueue_style( 'wcmd_optionsframework', WCMD_OPTIONS_FRAMEWORK_DIRECTORY . 'assets/css/wcmd_optionsframework.css', array(),  Wcmd_Options_Framework::VERSION );
			
			wp_enqueue_style( 'wcmd_bootflat', WCMD_OPTIONS_FRAMEWORK_DIRECTORY . 'assets/css/site.min.css', array(),  Wcmd_Options_Framework::VERSION );
			wp_enqueue_style( 'wp-color-picker' );
	}

	/**
	* Loads the required javascript
	*
	* @since 1.7.0
	*/
	function enqueue_admin_scripts( $hook ) {

		if ( $this->options_screen != $hook )
	        return;

	    $options = get_option( 'wcmd_options' );

	    wp_enqueue_script( 'wcmd-bootstrap', 'https://maxcdn.bootstrapcdn.com/bootstrap/3.3.7/js/bootstrap.min.js', array( 'jquery'), Wcmd_Options_Framework::VERSION );

	    // Enqueue icon-picker js
			wp_enqueue_script( 'icon-picker', WCMD_OPTIONS_FRAMEWORK_DIRECTORY . 'assets/js/jquery.fonticonpicker.min.js', array( 'jquery' ), Wcmd_Options_Framework::VERSION );

      // Enqueue SweetAlert2 Style
      wp_enqueue_style( 'Sweetalert2-css', WCMD_OPTIONS_FRAMEWORK_DIRECTORY . 'assets/css/sweetalert2.min.css', Wcmd_Options_Framework::VERSION );

      // Enqueue SweetAlert2 JS
      wp_enqueue_script( 'Sweetalert2-js', WCMD_OPTIONS_FRAMEWORK_DIRECTORY . 'assets/js/sweetalert2.all.min.js', array( 'jquery'), Wcmd_Options_Framework::VERSION );

		  wp_enqueue_style( 'select2-style', WCMD_OPTIONS_FRAMEWORK_DIRECTORY . 'assets/css/select2.min.css', Wcmd_Options_Framework::VERSION );

		  // Enqueue custom option panel JS
		  wp_enqueue_script( 'Select2-js', WCMD_OPTIONS_FRAMEWORK_DIRECTORY . 'assets/js/select2.full.js', array( 'jquery'), Wcmd_Options_Framework::VERSION );

		  wp_enqueue_script( 'options-custom', WCMD_OPTIONS_FRAMEWORK_DIRECTORY . 'assets/js/options-custom.js', array( 'jquery', 'wp-color-picker', 'icon-picker', 'Select2-js', 'Sweetalert2-js' ), Wcmd_Options_Framework::VERSION );

		  $option_var = array(
          'reset_text'              => __('Click OK to reset. Any settings will be lost!', 'wcmd'),
					'loader_text'             => __('Please Wait', 'wcmd'),
          'empty_api_key_error'     => __('Looks like an invalid API Key', 'wc_mailchimp_discount'),
          'empty_api_list_id'       => __('Please enter valid API Key and List Id/Audience Id', 'wc_mailchimp_discount'),
          'invalid_api_key'         => __('The API Key what you are using seems like an invalid key', 'wc_mailchimp_discount'),
          'valid_api_key'           => __('Seems like everything is fine here', 'wc_mailchimp_discount'),
          'use_predefined_coupon'  => isset($options['wcmd_use_coupons_from']) ? $options['wcmd_use_coupons_from'] : 'no',
					'ajax_nonce'	            => wp_create_nonce('wcmd_database_updater'),
          'wcmd_verify_api_nonce' => wp_create_nonce('wcmd-verify-api-nonce'),
          'wcmd_verify_webhook_nonce' => wp_create_nonce('wcmd-verify-webhook-nonce'),
          'wcmd_lists_nonce'  => wp_create_nonce('wcmd-lists-nonce'),
					'ajaxurl'  	 	            =>  admin_url( 'admin-ajax.php' ),
			);

		wp_localize_script( 'options-custom', 'wcmdOption' , $option_var );

		// Inline scripts from options-interface.php
		add_action( 'admin_head', array( $this, 'wcmd_of_admin_head' ) );
	}

	function wcmd_of_admin_head() {
		// Hook to add custom scripts
		do_action( 'wcmd_optionsframework_custom_scripts' );
	}

	private function mg_plugin_installed() {
		
		$plugin_list = array();

		if( class_exists('woocommerce') ) 
			array_push($plugin_list, 'woocommerce');
		
		if( class_exists('Easy_Digital_Downloads') )
			array_push($plugin_list, 'EDD');

		if( class_exists('Cornerstone_Plugin') )
			array_push($plugin_list, 'CornerStone');

		return $plugin_list;

	}

	private function mg_get_more_items() {

		$items = get_transient('api_list_of_items');

		$plugins_installed = $this->mg_plugin_installed();

		if( !$items )
			$items = $this->mg_api_fetch_items();

		$html = '';

		if( is_array($items) && !empty($items) ) {
			
			$html .= '<h5>'.__('Check Our Other Plugins', 'wcmd').'</h5>';
			$html .= '<div class="lSAction prev"><a class="lSPrev"></a></div>';
			$html .= '<ul id="owl-carousel">';


			//echo PLUGIN_NAME; //WooCommerce Mailchimp Discount
			foreach( $items as $key => $item ) {
				if( strtolower($item->title) !== strtolower(PLUGIN_NAME) ) {
					if( $item->product_image !== false ) {							
						if( strpos(strtolower($item->category), 'wordpress') !== false ) {
							if( is_array($plugins_installed) && !empty($plugins_installed) ) {
								foreach( $plugins_installed as $plugins_install ) {
									$plugin_install = strtolower($plugins_install);
									if( strpos(strtolower($item->category), $plugin_install) !== false ) {
										$html .= '<li class="item"><a alt="'.$item->title.'" href="'.$item->link.'" target="_blank"><img src="'.$item->product_image.'"></a></li>';
									}
								}
							}
						}
					}
				}
			}

			$html .= '</ul>';
			$html .= '<div class="lSAction next"><a class="lSNext"></a></div>';
		}

			echo $html;
	}

	private function mg_api_fetch_items() {

		$site_name = 'https://www.magnigenie.com';
		$remoteLink = $site_name.'/wp-json/magni-api-server/v1/items';
		
		$data = wp_remote_get($remoteLink);
		$items = array();

		if( is_array($data) && isset($data['body']) ) {
			$items = $data['body'];
			$items = json_decode($items);
			set_transient('api_list_of_items', $items, 60 * 60 * 24);		
		}
		return $items;
	}

	/**
     * Builds out the options panel.
     *
	 * If we were using the Settings API as it was intended we would use
	 * do_settings_sections here.  But as we don't want the settings wrapped in a table,
	 * we'll call our own custom wcmd_optionsframework_fields.  See options-interface.php
	 * for specifics on how each individual field is generated.
	 *
	 * Nonces are provided using the settings_fields()
	 *
     * @since 1.7.0
     */
	 function options_page() { ?>

		<div id="wcmd_optionsframework-wrap" class="wrap">

			<?php $menu = $this->menu_settings(); ?>
			<h2><?php echo esc_html( $menu['page_title'] ); ?></h2>

	    <h2 class="nav-tab-wrapper">
	        
	    </h2>

      <div class="wcmd-options-settings-wrapper">
        <?php settings_errors( 'options-framework' ); ?>

        <!--Navigation Tabs Starts Here -->
        <div class="mg-navtabs-wrapper">
          <?php echo Wcmd_Options_Framework_Interface::wcmd_optionsframework_tabs(); ?>
        </div>
        <!-- Navigation Tabs Ends Here -->

        <div class="mg-settings-panel-right">
          <div id="wcmd_optionsframework-metabox" class="metabox-holder">
            <div id="wcmd_optionsframework" class="postbox">
              <form action="options.php" id="wcmd_form_settings" method="post">
                <?php settings_fields( 'wcmd_optionsframework' ); ?>
                <?php Wcmd_Options_Framework_Interface::wcmd_optionsframework_fields(); /* Settings */ ?>
                <input type="submit" class="reset-button wcmd-reset-button button-secondary" style="display: none;" name="reset" value="<?php esc_attr_e( 'Restore Defaults', 'wc_mailchimp_discount' ); ?>" />
                <div class="clear"></div>
              </form>
            </div> <!-- / #container -->
          </div>

          <div class="mg-options-submit-wrap">
          <div id="wcmd_optionsframework-submit">
            <input type="submit" class="button-primary save-settings" name="update" value="<?php esc_attr_e( 'Save Changes', 'wc_mailchimp_discount' ); ?>" />
            <input type="submit" class="button-secondary reset-settings" name="update" value="<?php esc_attr_e( 'Restore Defaults', 'wc_mailchimp_discount' ); ?>"  />
            <div class="wcmd-floating-menus save-settings">
              <div class="wcmd-menu-quick-save wcmd-floating-button">
                <i class="wcmd-quicksave-icon"></i>
                  <input type="submit" class="wcmd-quick-save-btn" name="update" value="<?php esc_attr_e( 'SAVE', 'wc_mailchimp_discount' ); ?>" />
              </div><!-- / wcmd-menu-quick-save -->
              <div class="clear"></div>
            </div> <!-- / wcmd-floating-menus-->
            <div class="clear"></div>
          </div>

        </div>
          
        </div>
        

    
        
      </div>
      
			<?php do_action( 'wcmd_optionsframework_after' ); ?>

		</div> <!-- / .wrap -->

	<?php
	}

	/**
	 * Validate Options.
	 *
	 * This runs after the submit/reset button has been clicked and
	 * validates the inputs.
	 *
	 * @uses $_POST['reset'] to restore default options
	 */
	function validate_options( $input ) {

		/*
		 * Restore Defaults.
		 *
		 * In the event that the user clicked the "Restore Defaults"
		 * button, the options defined in the theme's options.php
		 * file will be added to the option for the active theme.
		 */


		if ( isset( $_POST['reset'] ) ) {
			add_settings_error( 'options-framework', 'restore_defaults', __( 'Default options restored.', 'wc_mailchimp_discount' ), 'updated fade in' );
			return $this->get_default_values();
		}

		/*
		 * Update Settings
		 *
		 * This used to check for $_POST['update'], but has been updated
		 * to be compatible with the theme customizer introduced in WordPress 3.4
		 */

		$clean = array();
		$options = & Wcmd_Options_Framework::_wcmd_optionsframework_options();
		
		foreach ( $options as $option ) {

			if ( ! isset( $option['id'] ) ) {
				continue;
			}

			if ( ! isset( $option['type'] ) ) {
				continue;
			}

			$id = preg_replace( '/[^a-zA-Z0-9._\-]/', '', strtolower( $option['id'] ) );

			// Set checkbox to false if it wasn't sent in the $_POST
			if ( 'checkbox' == $option['type'] && ! isset( $input[$id] ) ) {
				$input[$id] = false;
			}

			// Set each item in the multicheck to false if it wasn't sent in the $_POST
			if ( 'multicheck' == $option['type'] && ! isset( $input[$id] ) ) {
				foreach ( $option['options'] as $key => $value ) {
					$input[$id][$key] = false;
				}
			}

			// For a value to be submitted to database it must pass through a sanitization filter
			if ( has_filter( 'wcmd_of_sanitize_' . $option['type'] ) ) {
				$clean[$id] = apply_filters( 'wcmd_of_sanitize_' . $option['type'], $input[$id], $option );
			}
		}

		// Hook to run after validation
		do_action( 'wcmd_optionsframework_after_validate', $clean );

		return $clean;
	}

	/**
	 * Display message when options have been saved
	 */

	function save_options_notice() {
		add_settings_error( 'options-framework', 'save_options', PLUGIN_NAME . __( ' Settings Saved.', 'wc_mailchimp_discount' ), 'updated fade in' );
	}

	/**
	 * Get the default values for all the theme options
	 *
	 * Get an array of all default values as set in
	 * options.php. The 'id','std' and 'type' keys need
	 * to be defined in the configuration array. In the
	 * event that these keys are not present the option
	 * will not be included in this function's output.
	 *
	 * @return array Re-keyed options configuration array.
	 *
	 */

	function get_default_values() {
		$output = array();
		$config = & Wcmd_Options_Framework::_wcmd_optionsframework_options();
		foreach ( (array) $config as $option ) {
			if ( ! isset( $option['id'] ) ) {
				continue;
			}
			if ( ! isset( $option['std'] ) ) {
				continue;
			}
			if ( ! isset( $option['type'] ) ) {
				continue;
			}
			if ( has_filter( 'wcmd_of_sanitize_' . $option['type'] ) ) {
				$output[$option['id']] = apply_filters( 'wcmd_of_sanitize_' . $option['type'], $option['std'], $option );
			}
		}
		return $output;
	}

	/**
	 * Add options menu item to admin bar
	 */

	function wcmd_optionsframework_admin_bar() {

		$menu = $this->menu_settings();

		global $wp_admin_bar;

		if ( 'menu' == $menu['mode'] ) {
			$href = admin_url( 'admin.php?page=' . $menu['menu_slug'] );
		} else {
			$href = admin_url( 'themes.php?page=' . $menu['menu_slug'] );
		}

		$args = array(
			'parent' 	=> 'appearance',
			'id' 			=> 'wcmd_of_theme_options',
			'title' 	=> $menu['menu_title'],
			'href' 		=> $href
		);

		$wp_admin_bar->add_menu( apply_filters( 'wcmd_optionsframework_admin_bar', $args ) );
	}

}